#!/usr/bin/ruby
# @file   ndk_clientmanager.rb
# @author K.S.
#
# $Date: 2004/04/19 10:23:48 $
# $Id: ndk_manager.rb,v 1.3 2004/04/19 10:23:48 ko1 Exp $
#
# Create : K.S. 04/04/17 17:00:44
#

require 'rice/irc'
require 'ndk_config'
require 'ndk_client'

module Nadoka
  Cmd = ::RICE::Command
  Rpl = ::RICE::Reply

  require 'ndk_bot'

  class NDK_Error < Exception
  end
  
  class NDK_QuitClient < NDK_Error
  end
  
  class NDK_State
    def initialize
      @current_nick = nil
      @current_channels = []
    end
    attr_accessor :current_nick
    attr_reader   :current_channels
    def nick
      @current_nick
    end
  end
  
  class NDK_Manager
    def initialize rc
      @rc = rc
      reload_config
      @clients= []

      sv, port = @config.server_list.shift
      @config.server_list.push [sv, port]
      
      @server = ::RICE::Connection.new(sv, port)
      @state  = NDK_State.new

    end
    attr_reader :state

    def reload_config
      @config = NDK_Config.new(@rc)
      
      Dir.glob(@config.plugins_dir + '/*.nb'){|file|
        load file
      }
      
      @bots   = @config.bots.map{|bot|
        bot = eval(bot.to_s).new(self, @config)
        bot if bot.kind_of? NDK_Bot
      }.compact
    end

    def start_server_thread
      @server.regist{|rq, wq|
        Thread.stop
        begin
          server_main_thread rq,wq
        rescue Object
          @config.slog $!
          @config.slog $!.backtrace.join(' ')
          @server_thread.kill if @server_thread.alive?
        end
      }
      @server_thread = Thread.new{
        begin
          @config.slog "Server connection to #{@server.server}:#{@server.port}"
          @server.start(1)
        rescue RICE::Connection::Closed
          sv, port = @config.server_list.shift
          @config.server_list.push [sv, port]
          @server.server = sv
          @server.port   = port
          retry
        rescue Object
          @config.slog $!
          @config.slog $!.backtrace.join(' ')
        end
      }
    end

    def server_main_thread rq, wq
      # login
      if @config.server_passwd
        send_to_server Cmd.pass(@config.server_passwd)
      end
      send_to_server Cmd.nick(@config.nick)
      @state.current_nick = @config.nick
      
      send_to_server Cmd.user(@config.user,
                              @config.hostname, @config.servername,
                              @config.realname)
      # wait welcome message
      while q = rq.pop
        @config.dlog "[<S] #{q}"
        if q.command == '001'
          break
        end
      end
      
      # join to default channels
      @config.default_channels.each{|ch|
        send_to_server Cmd.join(ch)
      }
      
      # loop
      while q = rq.pop
        @config.dlog "[<S] #{q}"
        
        case q.command
        when 'PING'
          send_to_server Cmd.pong(q.params[0])
          next
        when 'PRIVMSG'
          if ctcp_message?(q.params[1])
            ctcp_message(q)
          end
        when 'JOIN'
          if about_me?(q)
            @state.current_channels.push q.params[0]
          end
        when 'PART'
          if about_me?(q)
            @state.current_channels.delete q.params[0]
          end
        when 'NICK'
          if about_me?(q)
            @state.current_nick = q.params[0]
          end
        else
          
        end
        send_to_clients q
        send_to_bot q
        logging q
      end
    end
    
    def start_clients_thread
      @clients_thread = Thread.new{
        @cserver = TCPServer.new(@config.client_server_port)
        @config.slog "Open Client Server Port: #{@cserver.addr.join(' ')}"

        begin
          while true
            # wait for client connections
            Thread.start(@cserver.accept){|cc|
              client = NDK_Client.new(@config, cc, self)
              @clients << client
              client.start
              @clients.delete client
            }
          end
        rescue
        ensure
          @server_thread.kill if @server_thread.alive?
        end
      }
    end

    def about_me?(msg)
      /^#{@state.current_nick}!/ =~ msg.prefix
    end
    
    def start
      start_server_thread
      start_clients_thread

      begin
        @server_thread.join
      rescue Interrupt
        @server_thread.kill  if @server_thread.alive?
        @clients_thread.kill if @clients_thread.alive?
      ensure
        @cserver.close
      end
    end
    
    def send_to_server msg
      @config.dlog "[S>] #{msg}"
      @server << msg
    end
    
    def send_to_clients msg
      @clients.each{|cl|
        cl << msg
      }
    end

    ::RICE::Command.regist_command('NADOKA')
    
    # "client to server" handling
    def send_from_client msg, from
      case msg.command
      when 'NADOKA'
        case msg.params[0].to_s.upcase
        when 'QUIT'
          exit
        when 'RESTART'
          from << Cmd.notice(@state.current_nick, 'RESTART: unsupport. sorry')
        when 'RELOAD'
          reload_config
          from << Cmd.notice(@state.current_nick, "configuration is reloaded")
        when 'RELOADBOT'
          
        when 'HELP'
          from << Cmd.notice(@state.current_nick, 'available: QUIT, RESTART, RELOAD, RELOADBOT, HELP')
        else
          from << Cmd.notice(@state.current_nick, 'No such command. Use /NADOKA HELP.')
        end
        return
      when 'QUIT'
        raise NDK_QuitClient
      when 'PRIVMSG', 'NOTICE'
        send_to_clients_otherwise msg, from
      else
      end
      send_to_server msg
    end

    def send_to_clients_otherwise msg, elt
      @clients.each{|cl|
        if cl != elt
          cl.add_prefix(msg) unless msg.prefix
          cl << msg
        end
      }
      send_to_bot msg
      logging msg
    end
    
    def join_channel ch
      send_to_server Cmd.topic(ch)
      send_to_server Cmd.names(ch)
    end

    def ctcp_message? arg
      arg[0] == 1
    end
    
    def ctcp_message msg
      if /\001(.+)\001/ =~ msg.params[1]
        ctcp_cmd = $1
        case ctcp_cmd
        when 'VERSION'
          send_to_server Cmd.notice(sender_of(msg),
          "\001VERSION nadoka Ver.#{NDK_Version}" +
          " with Ruby #{RUBY_VERSION} #{RUBY_PLATFORM}\001")
        when 'TIME'
          send_to_server Cmd.notice(sender_of(msg), "\001TIME #{Time.now}\001")
        end
      end
    end
    
    def sender_of msg
      if /^([^!]+)\!?/ =~ msg.prefix.to_s
        $1
      else
        @state.nick
      end
    end
    
    def send_to_bot msg
      selector = msg.command.downcase
      if /^\d+$/ =~ selector
        selector = RICE::Reply::Replies_num_to_name[selector]
      end
      @bots.each{|bot|
        break unless bot.__send__('on_'+selector, sender_of(msg), *msg.params)
      }
    end

    def logging msg
      user = sender_of(msg)
      ch   = msg.params[0]
      case msg.command
      when 'PRIVMSG'
        str = "<#{ch}:#{user}> #{msg.params[1]}"
        @config.clog ch, str
        
      when 'NOTIFY'
        str = "<#{ch}:#{user}> #{msg.params[1]}"
        @config.clog ch, str
        
      when /^\d+/
        str = msg.command + ' ' + msg.params.join(' ')
        @config.slog str
      end
    end
  end
end


